import { NextRequest, NextResponse } from 'next/server';
import { openDB } from '@/src/lib/db';

// GET /api/projects/[id] - פרויקט בודד
export async function GET(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const db = await openDB();
        const { id } = await params;

        const project = await new Promise((resolve, reject) => {
            db.get(`
                SELECT 
                    p.*,
                    c.name as customer_name,
                    c.company as customer_company,
                    c.email as customer_email,
                    c.phone as customer_phone
                FROM projects p
                LEFT JOIN customers c ON p.customer_id = c.id
                WHERE p.id = ?
            `, [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        if (!project) {
            return NextResponse.json(
                { error: 'פרויקט לא נמצא' },
                { status: 404 }
            );
        }

        // המרת JSON fields
        const formattedProject = {
            ...project,
            assigned_team: project.assigned_team ? (typeof project.assigned_team === 'string' ? JSON.parse(project.assigned_team) : project.assigned_team) : [],
            budget: project.budget || 0,
            progress: project.progress || 0
        };

        return NextResponse.json(formattedProject);

    } catch (error) {
        console.error('Error fetching project:', error);
        return NextResponse.json(
            { error: 'שגיאה בטעינת הפרויקט' },
            { status: 500 }
        );
    }
}

// PUT /api/projects/[id] - עדכון פרויקט
export async function PUT(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const db = await openDB();
        const { id } = await params;
        const body = await request.json();

        const {
            title,
            description,
            customer_id,
            status,
            start_date,
            end_date,
            budget,
            progress,
            assigned_team,
            notes
        } = body;

        // בדיקה שהפרויקט קיים
        const existingProject = await new Promise((resolve, reject) => {
            db.get('SELECT id FROM projects WHERE id = ?', [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        if (!existingProject) {
            return NextResponse.json(
                { error: 'פרויקט לא נמצא' },
                { status: 404 }
            );
        }

        const updateQuery = `
            UPDATE projects SET
                title = COALESCE(?, title),
                description = COALESCE(?, description),
                customer_id = COALESCE(?, customer_id),
                status = COALESCE(?, status),
                start_date = COALESCE(?, start_date),
                end_date = COALESCE(?, end_date),
                budget = COALESCE(?, budget),
                progress = COALESCE(?, progress),
                assigned_team = COALESCE(?, assigned_team),
                notes = COALESCE(?, notes),
                updated_at = CURRENT_TIMESTAMP
            WHERE id = ?
        `;

        await new Promise((resolve, reject) => {
            db.run(updateQuery, [
                title,
                description,
                customer_id,
                status,
                start_date,
                end_date,
                budget,
                progress,
                assigned_team ? JSON.stringify(assigned_team) : undefined,
                notes,
                id
            ], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ changes: this.changes });
                }
            });
        });

        return NextResponse.json({
            message: 'פרויקט עודכן בהצלחה'
        });

    } catch (error) {
        console.error('Error updating project:', error);
        return NextResponse.json(
            { error: 'שגיאה בעדכון הפרויקט' },
            { status: 500 }
        );
    }
}

// DELETE /api/projects/[id] - מחיקת פרויקט
export async function DELETE(
    request: NextRequest,
    { params }: { params: { id: string } }
) {
    try {
        const db = await openDB();
        const { id } = await params;

        // בדיקה שהפרויקט קיים
        const existingProject = await new Promise((resolve, reject) => {
            db.get('SELECT id FROM projects WHERE id = ?', [id], (err, row) => {
                if (err) {
                    reject(err);
                } else {
                    resolve(row);
                }
            });
        });

        if (!existingProject) {
            return NextResponse.json(
                { error: 'פרויקט לא נמצא' },
                { status: 404 }
            );
        }

        await new Promise((resolve, reject) => {
            db.run('DELETE FROM projects WHERE id = ?', [id], function (err) {
                if (err) {
                    reject(err);
                } else {
                    resolve({ changes: this.changes });
                }
            });
        });

        return NextResponse.json({
            message: 'פרויקט נמחק בהצלחה'
        });

    } catch (error) {
        console.error('Error deleting project:', error);
        return NextResponse.json(
            { error: 'שגיאה במחיקת הפרויקט' },
            { status: 500 }
        );
    }
}
